"""Form definitions for the testing platform."""
from __future__ import annotations

from flask_wtf import FlaskForm
from flask_wtf.file import FileAllowed, FileField, FileRequired
from wtforms import (
    BooleanField,
    FieldList,
    FormField,
    HiddenField,
    IntegerField,
    PasswordField,
    SelectField,
    SelectMultipleField,
    StringField,
    SubmitField,
    TextAreaField,
)
from wtforms.validators import DataRequired, EqualTo, InputRequired, Length, NumberRange, Optional


class LoginForm(FlaskForm):
    username = StringField(
        "Имя пользователя",
        validators=[DataRequired(), Length(max=64)],
    )
    password = PasswordField("Пароль", validators=[DataRequired()])
    remember_me = BooleanField("Запомнить меня")
    submit = SubmitField("Войти")


class UserCreateForm(FlaskForm):
    username = StringField("Имя пользователя", validators=[DataRequired(), Length(max=64)])
    last_name = StringField("Фамилия", validators=[DataRequired(), Length(max=128)])
    first_name = StringField("Имя", validators=[DataRequired(), Length(max=128)])
    middle_name = StringField("Отчество", validators=[Optional(), Length(max=128)])
    department = StringField("Отдел", validators=[DataRequired(), Length(max=128)])
    password = PasswordField(
        "Пароль",
        validators=[DataRequired(), Length(min=6)],
    )
    role = SelectField(
        "Роль",
        choices=[("user", "Пользователь"), ("admin", "Администратор")],
        validators=[DataRequired()],
    )
    submit = SubmitField("Создать")


class UserUpdateForm(FlaskForm):
    last_name = StringField("Фамилия", validators=[DataRequired(), Length(max=128)])
    first_name = StringField("Имя", validators=[DataRequired(), Length(max=128)])
    middle_name = StringField("Отчество", validators=[Optional(), Length(max=128)])
    department = StringField("Отдел", validators=[DataRequired(), Length(max=128)])
    role = SelectField(
        "Роль",
        choices=[("user", "Пользователь"), ("admin", "Администратор")],
        validators=[DataRequired()],
    )
    submit = SubmitField("Обновить")


class UserProfileForm(FlaskForm):
    last_name = StringField("Фамилия", validators=[DataRequired(), Length(max=128)])
    first_name = StringField("Имя", validators=[DataRequired(), Length(max=128)])
    middle_name = StringField("Отчество", validators=[Optional(), Length(max=128)])
    department = StringField("Отдел", validators=[DataRequired(), Length(max=128)])
    submit = SubmitField("Сохранить")


class AdminPasswordResetForm(FlaskForm):
    new_password = PasswordField(
        "Новый пароль",
        validators=[DataRequired(), Length(min=6)],
    )
    confirm_password = PasswordField(
        "Подтверждение",
        validators=[DataRequired(), EqualTo("new_password", message="Пароли должны совпадать")],
    )
    submit = SubmitField("Сбросить пароль")


class PasswordChangeForm(FlaskForm):
    current_password = PasswordField("Текущий пароль", validators=[DataRequired()])
    new_password = PasswordField("Новый пароль", validators=[DataRequired(), Length(min=6)])
    confirm_password = PasswordField(
        "Подтверждение",
        validators=[DataRequired(), EqualTo("new_password", message="Пароли должны совпадать")],
    )
    submit = SubmitField("Изменить пароль")


class AnswerForm(FlaskForm):
    answer_text = StringField("Ответ", validators=[DataRequired(), Length(max=512)])
    is_correct = BooleanField("Правильный ответ")


class QuestionForm(FlaskForm):
    question_text = TextAreaField("Вопрос", validators=[DataRequired()])
    question_type = SelectField(
        "Тип вопроса",
        choices=[("single", "Одиночный выбор"), ("multiple", "Множественный выбор")],
        validators=[DataRequired()],
    )
    answers = FieldList(FormField(AnswerForm), min_entries=2)


class GradeCriteriaForm(FlaskForm):
    grade = HiddenField(validators=[DataRequired()])
    max_errors = IntegerField(
        "Максимум ошибок",
        validators=[InputRequired(), NumberRange(min=0)],
    )


class TestForm(FlaskForm):
    title = StringField("Название теста", validators=[DataRequired(), Length(max=128)])
    description = TextAreaField("Описание", validators=[Optional()])
    is_active = BooleanField("Активен", default=True)
    shuffle_questions = BooleanField("Перемешивать вопросы", default=False)
    shuffle_answers = BooleanField("Перемешивать ответы", default=False)
    time_limit_minutes = IntegerField(
        "Лимит времени (минуты)",
        validators=[Optional(), NumberRange(min=1, max=1440)],
    )
    questions = FieldList(FormField(QuestionForm), min_entries=0)
    criteria = FieldList(FormField(GradeCriteriaForm), min_entries=0)
    submit = SubmitField("Сохранить")


class TestAssignmentForm(FlaskForm):
    assigned_to_all = BooleanField("Назначить всем пользователям")
    users = SelectMultipleField("Пользователи", coerce=int, validators=[Optional()])
    departments = SelectMultipleField("Отделы", coerce=str, validators=[Optional()])
    submit = SubmitField("Назначить")


class TestStartForm(FlaskForm):
    submit = SubmitField("Начать тест")


class TestSubmitForm(FlaskForm):
    submit = SubmitField("Завершить тест")


class TestImportForm(FlaskForm):
    document = FileField(
        "Документ Word",
        validators=[
            FileRequired(message="Выберите файл .docx"),
            FileAllowed(["docx"], message="Формат файла должен быть .docx"),
        ],
    )
    submit = SubmitField("Импортировать")
